----------------------------------------------------------------------------------
-- des56salt.vhd
--
-- Copyright (C) 2006 Michael Poppitz
-- 
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or (at
-- your option) any later version.
--
-- This program is distributed in the hope that it will be useful, but
-- WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along
-- with this program; if not, write to the Free Software Foundation, Inc.,
-- 51 Franklin St, Fifth Floor, Boston, MA 02110, USA
--
----------------------------------------------------------------------------------
--
-- Details: http://www.sump.org/projects/password/
--
-- Implementation of des with salting used for UNIX crypt.
-- DES portion based on the description and tables
-- in "Mathematical Cryptology" by Wayne Patterson.
--
----------------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity des56salt IS
	 Port ( key : in std_logic_vector(1 to 56);
			  load : in std_logic;
			  salt : in std_logic_vector(1 to 12);
			  clock : in std_logic;
			  reset : in std_logic;
			  result : out std_logic_vector(1 to 64);
			  ready2 : out std_logic
    );

end des56salt;

architecture Behavioral of des56salt is

	type SBox is array (0 to 63) of std_logic_vector(4 downto 1);
	constant sbox1: SBox := (
		x"e", x"4", x"d", x"1", x"2", x"f", x"b", x"8",	x"3", x"a", x"6", x"c", x"5", x"9", x"0", x"7",
		x"0", x"f", x"7", x"4", x"e", x"2", x"d", x"1",	x"a", x"6", x"c", x"b", x"9", x"5", x"3", x"8",
		x"4", x"1", x"e", x"8", x"d", x"6", x"2", x"b",	x"f", x"c", x"9", x"7", x"3", x"a", x"5", x"0",
		x"f", x"c", x"8", x"2", x"4", x"9", x"1", x"7",	x"5", x"b", x"3", x"e", x"a", x"0", x"6", x"d"
	);
	constant sbox2: SBox := (
		x"f", x"1", x"8", x"e", x"6", x"b", x"3", x"4", x"9", x"7", x"2", x"d", x"c", x"0", x"5", x"a",
		x"3", x"d", x"4", x"7", x"f", x"2", x"8", x"e", x"c", x"0", x"1", x"a", x"6", x"9", x"b", x"5",
		x"0", x"e", x"7", x"b", x"a", x"4", x"d", x"1", x"5", x"8", x"c", x"6", x"9", x"3", x"2", x"f",
		x"d", x"8", x"a", x"1", x"3", x"f", x"4", x"2", x"b", x"6", x"7", x"c", x"0", x"5", x"e", x"9"
	);
	constant sbox3: SBox := (
		x"a", x"0", x"9", x"e", x"6", x"3", x"f", x"5", x"1", x"d", x"c", x"7", x"b", x"4", x"2", x"8",
		x"d", x"7", x"0", x"9", x"3", x"4", x"6", x"a", x"2", x"8", x"5", x"e", x"c", x"b", x"f", x"1",
		x"d", x"6", x"4", x"9", x"8", x"f", x"3", x"0", x"b", x"1", x"2", x"c", x"5", x"a", x"e", x"7",
		x"1", x"a", x"d", x"0", x"6", x"9", x"8", x"7", x"4", x"f", x"e", x"3", x"b", x"5", x"2", x"c"
	);
	constant sbox4: SBox := (
		x"7", x"d", x"e", x"3", x"0", x"6", x"9", x"a", x"1", x"2", x"8", x"5", x"b", x"c", x"4", x"f",
		x"d", x"8", x"b", x"5", x"6", x"f", x"0", x"3", x"4", x"7", x"2", x"c", x"1", x"a", x"e", x"9",
		x"a", x"6", x"9", x"0", x"c", x"b", x"7", x"d", x"f", x"1", x"3", x"e", x"5", x"2", x"8", x"4",
		x"3", x"f", x"0", x"6", x"a", x"1", x"d", x"8", x"9", x"4", x"5", x"b", x"c", x"7", x"2", x"e"
	);
	constant sbox5: SBox := (
		x"2", x"c", x"4", x"1", x"7", x"a", x"b", x"6", x"8", x"5", x"3", x"f", x"d", x"0", x"e", x"9",
		x"e", x"b", x"2", x"c", x"4", x"7", x"d", x"1", x"5", x"0", x"f", x"a", x"3", x"9", x"8", x"6",
		x"4", x"2", x"1", x"b", x"a", x"d", x"7", x"8", x"f", x"9", x"c", x"5", x"6", x"3", x"0", x"e",
		x"b", x"8", x"c", x"7", x"1", x"e", x"2", x"d", x"6", x"f", x"0", x"9", x"a", x"4", x"5", x"3"
	);
	constant sbox6: SBox := (
		x"c", x"1", x"a", x"f", x"9", x"2", x"6", x"8", x"0", x"d", x"3", x"4", x"e", x"7", x"5", x"b",
		x"a", x"f", x"4", x"2", x"7", x"c", x"9", x"5", x"6", x"1", x"d", x"e", x"0", x"b", x"3", x"8",
		x"9", x"e", x"f", x"5", x"2", x"8", x"c", x"3", x"7", x"0", x"4", x"a", x"1", x"d", x"b", x"6",
		x"4", x"3", x"2", x"c", x"9", x"5", x"f", x"a", x"b", x"e", x"1", x"7", x"6", x"0", x"8", x"d"
	);
	constant sbox7: SBox := (
		x"4", x"b", x"2", x"e", x"f", x"0", x"8", x"d", x"3", x"c", x"9", x"7", x"5", x"a", x"6", x"1",
		x"d", x"0", x"b", x"7", x"4", x"9", x"1", x"a", x"e", x"3", x"5", x"c", x"2", x"f", x"8", x"6",
		x"1", x"4", x"b", x"d", x"c", x"3", x"7", x"e", x"a", x"f", x"6", x"8", x"0", x"5", x"9", x"2",
		x"6", x"b", x"d", x"8", x"1", x"4", x"a", x"7", x"9", x"5", x"0", x"f", x"e", x"2", x"3", x"c"
	);
	constant sbox8: SBox := (
		x"d", x"2", x"8", x"4", x"6", x"f", x"b", x"1", x"a", x"9", x"3", x"e", x"5", x"0", x"c", x"7",
		x"1", x"f", x"d", x"8", x"a", x"3", x"7", x"4", x"c", x"5", x"6", x"b", x"0", x"e", x"9", x"2",
		x"7", x"b", x"4", x"1", x"9", x"c", x"e", x"2", x"0", x"6", x"a", x"d", x"f", x"3", x"5", x"8",
		x"2", x"1", x"e", x"7", x"4", x"a", x"8", x"d", x"f", x"c", x"9", x"0", x"3", x"5", x"6", x"b"
	);

	signal counter: integer range 0 to 15;

	signal keyReg : std_logic_vector(1 to 56);
	alias keyLeft : std_logic_vector(1 to 28) is keyReg (1 to 28);
	alias keyRight : std_logic_vector(1 to 28) is keyReg (29 to 56);

	signal roundKey : std_logic_vector(1 to 48); 

	signal msgReg : std_logic_vector(1 to 64);
	alias msgLeft : std_logic_vector(1 to 32) is msgReg (1 to 32);
	alias msgRight : std_logic_vector(1 to 32) is msgReg (33 to 64);

	signal expanded, salted, xored : std_logic_vector (1 to 48);
	signal reduced, permutated : std_logic_vector (1 to 32);

	signal shiftKeyBy1, zero: std_logic;

begin

	-- round counter
	process(clock, reset)
	begin

		if reset = '1' then
			counter <= 0;
			ready2 <= '0';

		elsif rising_edge(clock) then
			if load = '1' then
				counter <= 1;
			elsif counter = 15 then
				counter <= 0;
			else
				counter <= counter + 1;
			end if;

			if counter = 14 then
				ready2 <= '1';
			else
				ready2 <= '0';
			end if;
		end if;

	end process;

	-- key control
   process(clock, counter)
	begin
	
		if rising_edge(clock) then

			if load = '1' then
				-- modified PC-1 permutation with no "blind" inputs that would annoy the synthesizer
				-- first shift by one is also included
				keyReg <=
								   key(43) & key(36) & key(29) & key(22) & key(15) & key(8)
					& key(1)  & key(51) & key(44) & key(37) & key(30) & key(23) & key(16)
					& key(9)  & key(2)  & key(52) & key(45) & key(38) & key(31) & key(24)
					& key(17) & key(10) & key(3)  & key(53) & key(46) & key(39) & key(32)
					& key(50)
								 & key(49) & key(42) & key(35) & key(28) & key(21) & key(14)
					& key(7)  & key(55) & key(48) & key(41) & key(34) & key(27) & key(20)
					& key(13) & key(6)  & key(54) & key(47) & key(40) & key(33) & key(26)
					& key(19) & key(12) & key(5)  & key(25) & key(18) & key(11) & key(4)
					& key(56);

			elsif shiftKeyBy1 = '1' then
            keyLeft(1 to 28) <= keyLeft(2 to 28) & keyLeft(1);
            keyRight(1 to 28) <= keyRight(2 to 28) & keyRight(1);

			else -- shift key by 2
            keyLeft(1 to 28) <= keyLeft(3 to 28) & keyLeft(1 to 2);
            keyRight(1 to 28) <= keyRight(3 to 28) & keyRight(1 to 2);

			end if;
		end if;
				
		case counter is
			when  0 => shiftKeyBy1 <= '1';
			when  1 => shiftKeyBy1 <= '1';
			when  8 => shiftKeyBy1 <= '1';
			when 15 => shiftKeyBy1 <= '1';
			when others =>	shiftKeyBy1 <= '0';
		end case;

	end process;

	-- obtain round key by applying PC-2
   roundKey <=
		  keyReg(14) & keyReg(17) & keyReg(11) & keyReg(24) & keyReg(1)  & keyReg(5)  & keyReg(3)  & keyReg(28)
      & keyReg(15) & keyReg(6)  & keyReg(21) & keyReg(10) & keyReg(23) & keyReg(19) & keyReg(12) & keyReg(4)
      & keyReg(26) & keyReg(8)  & keyReg(16) & keyReg(7)  & keyReg(27) & keyReg(20) & keyReg(13) & keyReg(2)
      & keyReg(41) & keyReg(52) & keyReg(31) & keyReg(37) & keyReg(47) & keyReg(55) & keyReg(30) & keyReg(40)
      & keyReg(51) & keyReg(45) & keyReg(33) & keyReg(48) & keyReg(44) & keyReg(49) & keyReg(39) & keyReg(56)
      & keyReg(34) & keyReg(53) & keyReg(46) & keyReg(42) & keyReg(50) & keyReg(36) & keyReg(29) & keyReg(32);

	-- data control
	process(clock, load)
	begin
		if rising_edge(clock) then
			if load = '1' then
				msgReg <= (others=> '0');
			else
				-- an "if = 0 then else" won't be synthesized the way I want it, so its spelled out (for single 4-input-LUT)
				for i in 1 to 32 loop
					msgLeft(i) <= ((permutated(i) xor msgLeft(i)) and zero) or (msgRight(i) and not zero);
					msgRight(i) <= (msgRight(i) and zero) or ((permutated(i) xor msgLeft(i)) and not zero);
				end loop;
			end if;
		end if;
	end process;

	zero <= '1' when counter = 0 else '0';

	-- expand 32 bits of right message half to 48 bits
	expanded <=
		  (msgRight(32) & msgRight(1 to 5))
		& (msgRight( 4 to  9))
		& (msgRight( 8 to 13))
		& (msgRight(12 to 17))
		& (msgRight(16 to 21))
		& (msgRight(20 to 25))
		& (msgRight(24 to 29))
		& (msgRight(28 to 32)) & msgRight(1);

	-- permutate according to salt
	salted <=
		  ((expanded(25 to 36) and salt) or (expanded(1 to 12) and not salt))
		& expanded(13 to 24)
		& ((expanded(1 to 12) and salt) or (expanded(25 to 36) and not salt))
		& expanded(37 to 48);

	-- mix it with the round key
	xored <= salted xor roundKey;

	-- reduce to 32 bits using sboxes
	reduced <=
		  sbox1(conv_integer(xored( 1) & xored( 6) & xored( 2 to  5)))
		& sbox2(conv_integer(xored( 7) & xored(12) & xored( 8 to 11)))
		& sbox3(conv_integer(xored(13) & xored(18) & xored(14 to 17)))
		& sbox4(conv_integer(xored(19) & xored(24) & xored(20 to 23)))
		& sbox5(conv_integer(xored(25) & xored(30) & xored(26 to 29)))
		& sbox6(conv_integer(xored(31) & xored(36) & xored(32 to 35)))
		& sbox7(conv_integer(xored(37) & xored(42) & xored(38 to 41)))
		& sbox8(conv_integer(xored(43) & xored(48) & xored(44 to 47)));

	-- apply permutation P
	permutated <=
		  reduced(16) & reduced(7)  & reduced(20) & reduced(21)
		& reduced(29) & reduced(12) & reduced(28) & reduced(17)
		& reduced(1)  & reduced(15) & reduced(23) & reduced(26)
		& reduced(5)  & reduced(18) & reduced(31) & reduced(10)
		& reduced(2)  & reduced(8)  & reduced(24) & reduced(14)
		& reduced(32) & reduced(27) & reduced(3)  & reduced(9)
		& reduced(19) & reduced(13) & reduced(30) & reduced(6)
		& reduced(22) & reduced(11) & reduced(4)  & reduced(25);
		
	-- apply reverse IP (outdata is only valid when all rounds have completed)
	result <=
		  msgReg(40) & msgReg(8)  & msgReg(48) & msgReg(16) & msgReg(56) & msgReg(24) & msgReg(64) & msgReg(32)
		& msgReg(39) & msgReg(7)  & msgReg(47) & msgReg(15) & msgReg(55) & msgReg(23) & msgReg(63) & msgReg(31)
		& msgReg(38) & msgReg(6)  & msgReg(46) & msgReg(14) & msgReg(54) & msgReg(22) & msgReg(62) & msgReg(30)
		& msgReg(37) & msgReg(5)  & msgReg(45) & msgReg(13) & msgReg(53) & msgReg(21) & msgReg(61) & msgReg(29)
		& msgReg(36) & msgReg(4)  & msgReg(44) & msgReg(12) & msgReg(52) & msgReg(20) & msgReg(60) & msgReg(28)
		& msgReg(35) & msgReg(3)  & msgReg(43) & msgReg(11) & msgReg(51) & msgReg(19) & msgReg(59) & msgReg(27)
		& msgReg(34) & msgReg(2)  & msgReg(42) & msgReg(10) & msgReg(50) & msgReg(18) & msgReg(58) & msgReg(26)
		& msgReg(33) & msgReg(1)  & msgReg(41) & msgReg(9)  & msgReg(49) & msgReg(17) & msgReg(57) & msgReg(25);
end;
